using System.Text;

public class Base64EncodingStream : Stream
{
    private readonly Stream _innerStream;
    private readonly MemoryStream _base64BufferStream = new MemoryStream();
    private bool _endOfStream = false;

    public Base64EncodingStream(Stream innerStream)
    {
        _innerStream = innerStream ?? throw new ArgumentNullException(nameof(innerStream));
    }

    public override bool CanRead => true;
    public override bool CanSeek => false;
    public override bool CanWrite => false;
    public override long Length => throw new NotSupportedException();
    public override long Position
    {
        get => throw new NotSupportedException();
        set => throw new NotSupportedException();
    }

    public override void Flush() => throw new NotSupportedException();

    public override int Read(byte[] buffer, int offset, int count)
    {
        if (_endOfStream && _base64BufferStream.Length == 0)
            return 0; // End of stream

        // Fill the Base64 buffer if it's empty
        if (_base64BufferStream.Length == 0)
        {
            byte[] readBuffer = new byte[81920]; // 80 KB
            int bytesRead = _innerStream.Read(readBuffer, 0, readBuffer.Length);

            if (bytesRead > 0)
            {
                string base64Chunk = Convert.ToBase64String(readBuffer, 0, bytesRead);
                byte[] base64Bytes = Encoding.UTF8.GetBytes(base64Chunk);
                _base64BufferStream.Write(base64Bytes, 0, base64Bytes.Length);
                _base64BufferStream.Position = 0; // Reset position for reading
            }
            else
            {
                _endOfStream = true;
            }
        }

        // Read Base64 data from the buffer
        int bytesToCopy = Math.Min(count, (int)(_base64BufferStream.Length - _base64BufferStream.Position));
        int bytesCopied = _base64BufferStream.Read(buffer, offset, bytesToCopy);

        // If the Base64 buffer is exhausted, reset it
        if (_base64BufferStream.Position == _base64BufferStream.Length)
        {
            _base64BufferStream.SetLength(0);
            _base64BufferStream.Position = 0;
        }

        return bytesCopied;
    }

    public override long Seek(long offset, SeekOrigin origin) => throw new NotSupportedException();
    public override void SetLength(long value) => throw new NotSupportedException();
    public override void Write(byte[] buffer, int offset, int count) => throw new NotSupportedException();

    protected override void Dispose(bool disposing)
    {
        if (disposing)
        {
            _innerStream?.Dispose();
            _base64BufferStream?.Dispose();
        }
        base.Dispose(disposing);
    }
}
